from typing import Any

bl_info = {
    "name": "Marionette for Blender",
    "author": "Marionette ApS",
    "category": "Animation",
    "location": "View 3D > Tool Shelf > Marionette",
    "description": "Stream your Marionette animations directly into Blender.",
    "version": (33, 6),
    "warning": "",
    "blender": (3, 6, 0),
    "doc_url": "https://marionettexr.com/redirect-to-online-documentation",
    "tracker_url": "https://marionettexr.com/contact-us/",
}

import traceback

first_startup = "bpy" not in locals()
import bpy
import bpy.utils
import bpy.app.timers
import bpy.app.handlers
from bpy.app.handlers import persistent
from bpy.types import Panel
from .properties import Context, Scene
import importlib


def show_error(
    message: str = "", title: str = "Unable to load Marionette for Blender plugin!"
) -> None:
    def draw(self: Panel, context: Context) -> None:
        layout = self.layout
        msg_list = message.split("\n")

        for i, msg in enumerate(msg_list):
            row = layout.row(align=True)
            row.scale_y = 0.85
            row.label(text=msg)

    # Only show popup if the preferences window is open
    if len(bpy.context.window_manager.windows) > 1:
        bpy.context.window_manager.popup_menu(draw, title=title, icon="ERROR")


classes = []


@persistent  # type: ignore[misc]
def on_load() -> float | None:
    from .core import preset_storage

    preset_storage.read_from_file()
    return None  # Cancel the timer


@persistent  # type: ignore[misc]
def on_load_post(_scene: Scene) -> None:
    from .core import preset_storage
    from .operators import client_operators

    preset_storage.read_from_file()
    if client_operators.client is not None:
        client_operators.client.close()
        client_operators.client = None


@persistent  # type: ignore[misc]
def on_undo_post(scene: Scene) -> None:
    from .core import preset_storage
    from .operators import client_operators

    # Undoing/redoing might change the blender state. We need to make sure the global/shared state is in sync.
    preset_storage.write_to_file()

    if scene.mb_wants_to_connect and client_operators.client is None:
        bpy.ops.mb.client_start()  # type: ignore[attr-defined]
    elif not scene.mb_wants_to_connect and client_operators.client is not None:
        client_operators.client.close()
        client_operators.client = None


@persistent  # type: ignore[misc]
def on_redo_post(scene: Scene) -> None:
    from .core import preset_storage
    from .operators import client_operators

    # Undoing/redoing might change the blender state. We need to make sure the global/shared state is in sync.
    preset_storage.write_to_file()

    if scene.mb_wants_to_connect and client_operators.client is None:
        bpy.ops.mb.client_start()  # type: ignore[attr-defined]
    elif not scene.mb_wants_to_connect and client_operators.client is not None:
        client_operators.client.close()
        client_operators.client = None


@persistent  # type: ignore[misc]
def on_scene_changed(self: Scene, _: Any) -> None:
    from .core import preset_storage

    if self.mb_editing_preset_old_name != "":
        if not self.mb_control_rig:
            self.mb_editing_preset_old_name = ""


def register() -> None:
    global first_startup
    print("\n### Loading Marionette for Blender...")
    try:
        # If first startup of this plugin, load all modules normally
        # If reloading the plugin, use importlib to reload modules
        # This lets you do adjustments to the plugin on the fly without having to restart Blender
        from . import core
        from . import mapping
        from . import ui
        from . import operators
        from . import properties

        if not first_startup:
            importlib.reload(core)
            importlib.reload(mapping)
            importlib.reload(ui)
            importlib.reload(operators)
            importlib.reload(properties)
        import bpy

        global classes
        classes = [
            ui.MarionettePanel,
            ui.InfoPanel,
            ui.MB_UL_preset_bone_map,
            ui.MB_UL_presets,
            operators.client_operators.ClientStart,
            operators.client_operators.ClientStop,
            operators.preset_operators.PresetAdd,
            operators.preset_operators.PresetRemove,
            operators.preset_operators.PresetMove,
            operators.preset_operators.PickBone,
            operators.preset_operators.EditPreset,
            operators.preset_operators.EditPresetCancel,
            operators.preset_operators.EditPresetSave,
            operators.open_website.OpenWebsite,
            properties.MappingItem,
            properties.TPoseMappingItem,
            properties.PresetItem,
        ]

        # Register classes
        for cls in classes:
            try:
                bpy.utils.register_class(cls)
            except ValueError:
                print("Error: Failed to register class", cls)
                pass

        # Register all custom properties
        properties.register()

        # Load custom icons
        core.icon_manager.load_icons()

        bpy.app.timers.register(on_load)
        bpy.app.handlers.load_post.append(on_load_post)
        bpy.app.handlers.undo_post.append(on_undo_post)
        bpy.app.handlers.redo_post.append(on_redo_post)
        bpy.app.handlers.depsgraph_update_post.append(on_scene_changed)

        print("### Loaded Marionette for Blender.\n")
    except Exception:
        print("\nERROR: Marionette for Blender plugin failed to load:")
        trace = traceback.format_exc()
        print(trace)
        show_error(
            'Make sure you have the latest version available from "https://marionettexr.com".\n\n'
            + trace
        )
        return


def unregister() -> None:
    print("### Unloading Marionette for Blender...")
    from . import operators
    from . import properties
    from . import core

    bpy.app.handlers.load_post.remove(on_load_post)
    bpy.app.handlers.undo_post.remove(on_undo_post)
    bpy.app.handlers.redo_post.remove(on_redo_post)
    bpy.app.handlers.depsgraph_update_post.remove(on_scene_changed)
    if bpy.app.timers.is_registered(on_load):
        bpy.app.timers.unregister(on_load)

    if operators.client_operators.client is not None:
        operators.client_operators.client.close()
        operators.client_operators.client = None

    properties.unregister()
    # Unregister all classes
    for cls in reversed(classes):
        try:
            bpy.utils.unregister_class(cls)
        except RuntimeError as e:
            print(e)

    # Unload all custom icons
    try:
        core.icon_manager.unload_icons()
    except RuntimeError as e:
        print(e)
    print("### Unloaded Marionette for Blender.\n")
