from __future__ import annotations

from typing import (
    TypeVar,
    Generic,
    overload,
    Sequence,
    Iterator,
    MutableSequence,
    Any,
    Literal,
)

import bpy
import bpy.types
from bpy.props import (
    IntProperty,
    PointerProperty,
    BoolProperty,
    StringProperty,
    FloatVectorProperty,
    CollectionProperty,
)
from bpy.types import Object

from . import mapping

# region Typing
T = TypeVar("T")
U = TypeVar("U")


class bpy_prop_collection_idprop(Generic[T]):
    def find(self, key: str | None) -> int: ...  # type: ignore[empty-body]
    def foreach_get(
        self,
        attr: str,
        seq: MutableSequence[bool] | MutableSequence[int] | MutableSequence[float],
    ) -> None: ...
    def foreach_set(
        self,
        attr: str,
        seq: Sequence[bool] | Sequence[int] | Sequence[float],
    ) -> None: ...
    def get(self, key: str | None, default: U | None = None) -> T | U: ...  # type: ignore[empty-body]
    def items(self) -> list[tuple[str, T]]: ...  # type: ignore[empty-body]
    def keys(self) -> list[str]: ...  # type: ignore[empty-body]
    def values(self) -> list[T]: ...  # type: ignore[empty-body]
    @overload
    def __getitem__(self, key: int | str) -> T: ...
    @overload
    def __getitem__(self, key: slice) -> tuple[T]: ...
    def __getitem__(self, key: int | str | slice) -> T | tuple[T]: ...  # type: ignore[empty-body]
    @overload
    def __setitem__(self, key: int | str, value: T) -> None: ...
    @overload
    def __setitem__(self, key: slice, value: tuple[T]) -> None: ...
    def __setitem__(self, key: int | str | slice, value: T | tuple[T]) -> None: ...
    def __delitem__(self, key: int | str | slice) -> None: ...
    def __iter__(self) -> Iterator[T]: ...  # type: ignore[empty-body]
    def __next__(self) -> T: ...  # type: ignore[empty-body]
    def __len__(self) -> int: ...  # type: ignore[empty-body]
    def __contains__(self, key: str | tuple[str, ...]) -> bool: ...  # type: ignore[empty-body]
    def add(self) -> T: ...  # type: ignore[empty-body]
    def clear(self) -> None: ...
    def move(self, src_index: int, dst_index: int) -> None: ...
    def remove(self, index: int) -> None: ...


class Scene(bpy.types.Scene):
    mb_client_fps: int
    mb_control_rig: bpy.types.Object | None
    mb_mirror_selection: bool
    mb_presets: bpy_prop_collection_idprop[PresetItem]
    mb_selected_preset_index: int
    mb_editing_preset_is_new: bool
    mb_editing_preset_old_name: str  # "" indicates that we are not currently editing
    mb_editing_preset: PresetItem
    mb_editing_preset_bone_map_index: int
    mb_editing_preset_message: str
    mb_pose_before_editing: bpy_prop_collection_idprop[TPoseMappingItem]
    mb_wants_to_connect: bool


class Context(bpy.types.Context):
    scene: Scene


# endregion


Position = tuple[float, float, float]
Rotation = tuple[float, float, float, float]


class TPoseMappingItem(bpy.types.PropertyGroup):
    bone_name: StringProperty(  # type: ignore[valid-type]
        name="Control rig bone name",
        description="Name of the bone in the control rig",
        default="",
    )

    local_position: FloatVectorProperty(  # type: ignore[valid-type]
        name="Local Position", description="Position of the bone in the t-pose", size=3
    )
    local_rotation: FloatVectorProperty(  # type: ignore[valid-type]
        name="Local Rotation", description="Rotation of the bone in the t-pose", size=4
    )
    world_position: FloatVectorProperty(  # type: ignore[valid-type]
        name="World Position", description="Position of the bone in the t-pose", size=3
    )
    world_rotation: FloatVectorProperty(  # type: ignore[valid-type]
        name="World Rotation", description="Rotation of the bone in the t-pose", size=4
    )
    rotation_mode: StringProperty(  # type: ignore[valid-type]
        name="Rotation Mode",
        description="Rotation mode of the bone when the t-pose was saved",
    )

    def copy_from(self, other: "TPoseMappingItem") -> None:
        self.bone_name = other.bone_name
        self.local_position = other.local_position
        self.local_rotation = other.local_rotation
        self.world_position = other.world_position
        self.world_rotation = other.world_rotation
        self.rotation_mode = other.rotation_mode


class MappingItem(bpy.types.PropertyGroup):
    human_name: StringProperty(  # type: ignore[valid-type]
        name="Marionette rig bone name",
        description="Name of the bone in the Marionette default rig",
        default="",
    )
    bone_name: StringProperty(  # type: ignore[valid-type]
        name="Control rig bone name",
        description="Name of the bone in the control rig",
        default="",
    )
    copy_rotation: BoolProperty(  # type: ignore[valid-type]
        name="Copy rotation",
        description="Whether to copy the retargeted world rotation onto this bone on the control rig",
        default=True,
    )
    copy_position: BoolProperty(  # type: ignore[valid-type]
        name="Copy position",
        description="Whether to copy the retargeted world position onto this bone on the control rig",
        default=True,
    )

    def copy_from(self, other: "MappingItem") -> None:
        self.human_name = other.human_name
        self.bone_name = other.bone_name
        self.copy_rotation = other.copy_rotation
        self.copy_position = other.copy_position

    def error_message(self, context: Context) -> str | None:
        status = self.validate(context)
        if status == "OK":
            return None
        elif status == "REQUIRED":
            return None
        elif status == "INVALID":
            return f"Bone with name '{self.bone_name}' not found"
        elif status == "MULTIPLE":
            return f"'{self.bone_name}' has been assigned in multiple places"
        else:
            return None

    def validate(
        self, context: Context
    ) -> Literal["REQUIRED", "MULTIPLE", "INVALID", "OK"]:
        if self.bone_name == "":
            if self.human_name in mapping.required_bones:
                return "REQUIRED"
            else:
                return "OK"
        for item in context.scene.mb_editing_preset.bone_map:
            if self.human_name != item.human_name and self.bone_name == item.bone_name:
                return "MULTIPLE"
        if context.scene.mb_control_rig:
            for bone in context.scene.mb_control_rig.pose.bones:
                if self.bone_name == bone.name:
                    return "OK"
        return "INVALID"

    def icon(self, context: Context) -> str:
        assert context is not None
        status = self.validate(context)
        if status == "OK":
            return "NONE"
        elif status == "REQUIRED":
            return "ERROR"
        elif status == "INVALID":
            return "CANCEL"
        elif status == "MULTIPLE":
            return "CANCEL"
        else:
            assert False


class PresetItem(bpy.types.PropertyGroup):
    def on_left_leg_ik_changed(self, context: Context) -> None:
        item: MappingItem
        for item in self.bone_map:
            if item.human_name in ["LeftUpperLeg", "LeftLowerLeg"]:
                item.copy_rotation = not self.left_leg_ik
                item.copy_position = not self.left_leg_ik

    def on_right_leg_ik_changed(self, context: Context) -> None:
        item: MappingItem
        for item in self.bone_map:
            if item.human_name in ["RightUpperLeg", "RightLowerLeg"]:
                item.copy_rotation = not self.right_leg_ik
                item.copy_position = not self.right_leg_ik

    def on_left_arm_ik_changed(self, context: Context) -> None:
        item: MappingItem
        for item in self.bone_map:
            if item.human_name in ["LeftUpperArm", "LeftLowerArm"]:
                item.copy_rotation = not self.left_arm_ik
                item.copy_position = not self.left_arm_ik

    def on_right_arm_ik_changed(self, context: Context) -> None:
        item: MappingItem
        for item in self.bone_map:
            if item.human_name in ["RightUpperArm", "RightLowerArm"]:
                item.copy_rotation = not self.right_arm_ik
                item.copy_position = not self.right_arm_ik

    name: StringProperty(  # type: ignore[valid-type]
        name="Preset name", description="Name of the preset", default="Untitled"
    )
    bone_map: CollectionProperty(  # type: ignore[valid-type]
        type=MappingItem,
        name="Bone mappings",
        description="Collection of bone mappings in the preset",
    )
    left_leg_ik: BoolProperty(  # type: ignore[valid-type]
        name="Left Leg",
        description="Enables IK support for the left leg",
        default=False,
        update=on_left_leg_ik_changed,
    )
    right_leg_ik: BoolProperty(  # type: ignore[valid-type]
        name="Right Leg",
        description="Enables IK support for the right leg",
        default=False,
        update=on_right_leg_ik_changed,
    )
    left_arm_ik: BoolProperty(  # type: ignore[valid-type]
        name="Left Arm",
        description="Enables IK support for the left arm",
        default=False,
        update=on_left_arm_ik_changed,
    )
    right_arm_ik: BoolProperty(  # type: ignore[valid-type]
        name="Right Arm",
        description="Enables IK support for the right arm",
        default=False,
        update=on_right_arm_ik_changed,
    )
    tpose: CollectionProperty(  # type: ignore[valid-type]
        type=TPoseMappingItem,
        name="T-Pose",
        description="List of t-pose position and rotation for each bone in the armature",
    )

    def copy_from(self, other: "PresetItem") -> None:
        self.name = other.name
        self.left_arm_ik = other.left_arm_ik
        self.right_arm_ik = other.right_arm_ik
        self.left_leg_ik = other.left_leg_ik
        self.right_leg_ik = other.right_leg_ik
        self.bone_map.clear()
        for other_item in other.bone_map:
            self.bone_map.add().copy_from(other_item)
        self.tpose.clear()
        for other_item in other.tpose:
            self.tpose.add().copy_from(other_item)


def poll_control_rig(self: Scene, obj: Object) -> bool:
    return obj.type == "ARMATURE"


def on_control_rig_selection_updated(self: Scene, context: Context) -> None:
    context.scene.mb_editing_preset_old_name = ""


custom_properties: list[tuple[str, Any]] = [
    (
        "mb_has_initialized",
        BoolProperty(
            name="Marionette Has Initialized",
            description="Whether the add-on has been initialized",
            default=False,
        ),
    ),
    (
        "mb_client_fps",
        IntProperty(
            name="Client FPS",
            description="Rate at which messages are read/sent",
            default=60,
            min=1,
            max=120,
        ),
    ),
    (
        "mb_control_rig",
        PointerProperty(
            name="Control Rig",
            description="Select the armature with the control rig",
            type=bpy.types.Object,
            poll=poll_control_rig,
            update=on_control_rig_selection_updated,
        ),
    ),
    (
        "mb_mirror_selection",
        BoolProperty(
            name="Mirror Selection",
            description="Select both left and right bones at the same times when using the bone picker",
            default=True,
        ),
    ),
    (
        "mb_presets",
        CollectionProperty(
            name="Presets",
            description="List of all globally available presets",
            type=PresetItem,
        ),
    ),
    ("mb_selected_preset_index", IntProperty(name="Selected Preset Index")),
    (
        "mb_editing_preset_is_new",
        BoolProperty(
            name="Editing Preset New",
            description="Is the currently edited preset a brand new preset",
            default=False,
        ),
    ),
    (
        "mb_editing_preset_old_name",
        StringProperty(
            name="Editing Preset Name",
            description="What preset is currently being edited",
            default="",
        ),
    ),
    ("mb_editing_preset", PointerProperty(type=PresetItem)),
    (
        "mb_editing_preset_bone_map_index",
        IntProperty(name="Bone Mapping Index", default=0),
    ),
    (
        "mb_editing_preset_message",
        StringProperty(
            name="Editing Preset Message",
            description="Message that is displayed at the top of the edit preset panel",
            default="",
        ),
    ),
    (
        "mb_pose_before_editing",
        CollectionProperty(
            name="Pose Before Editing",
            description="Pose of the rig before editing began",
            type=TPoseMappingItem,
        ),
    ),
    (
        "mb_wants_to_connect",
        BoolProperty(
            name="Should Connect",
            description="Whether or not the client should try to connect",
            default=False,
        ),
    ),
]


def register() -> None:
    for name, value in custom_properties:
        setattr(bpy.types.Scene, name, value)


def unregister() -> None:
    for name, _ in custom_properties:
        delattr(bpy.types.Scene, name)
