import socket
import struct
import sys
import threading
import traceback
import weakref
from typing import cast, Final

from .message_handler import MessageHandler, ExecuteOnMainThreadFunction
from .preset_storage import Preset
from .stream import Stream

PROTOCOL_VERSION = 5


def _close(wants_to_stop: threading.Event, wake: threading.Event) -> None:
    wants_to_stop.set()
    wake.set()
    print("Client: Closing...")


class Client:
    def __init__(self, ip: str, port: int, preset: Preset) -> None:
        """
        Note: the message_handler will be passed to the background thread,
        and should not be modified outside of this class.
        """
        self.message_handler: Final = MessageHandler()
        self._thread: Final = threading.Thread(
            target=self._run, args=[ip, port, preset]
        )

        self._wants_to_stop: Final = threading.Event()
        self._wake: Final = threading.Event()
        self._wake.set()
        self._lock: Final = threading.Lock()
        self._is_connected = False

        self._finalizer: Final = weakref.finalize(
            self, _close, self._wants_to_stop, self._wake
        )
        self._thread.start()

    @property
    def is_connected(self) -> bool:
        with self._lock:
            return self._is_connected

    @property
    def closed(self) -> bool:
        return not self._finalizer.alive

    def close(self) -> None:
        self._finalizer()

    def _run(self, ip: str, port: int, preset: Preset) -> None:
        try:
            print("Client: Starting thread...")
            stream = Stream()
            while not self._wants_to_stop.is_set():
                print(f"Client: Connecting to {ip}:{port}...")
                try:
                    with socket.socket(
                        socket.AF_INET, socket.SOCK_STREAM
                    ) as connection:
                        connection.settimeout(15)
                        connection.connect((ip, port))
                        stream.socket = connection
                        self.message_handler.on_connected(
                            stream, PROTOCOL_VERSION, preset
                        )
                        with self._lock:
                            self._is_connected = True
                        print("Client: Connected.")
                        while True:
                            if self._wants_to_stop.is_set():
                                print(f"Client: Disconnecting from {ip}:{port}.")
                                break

                            if not self.message_handler.read_message(stream):
                                # Quit requested.
                                print("Client: Server requested quit.")
                                break
                        print("Client: Shutting down connection.")
                        connection.shutdown(socket.SHUT_RDWR)
                    print(f"Client: Connection to {ip}:{port} closed.")
                except (socket.error, struct.error, ConnectionError, TimeoutError):
                    print("Client: Connection error.")
                    print(traceback.format_exc(), file=sys.stderr)
                with self._lock:
                    self._is_connected = False
                print(f"Client: Message handler called.")
            print("Client: Thread exited.")
        except Exception as error:
            try:
                connection.shutdown(socket.SHUT_RDWR)
            except Exception:
                pass  # Ignored.
            with self._lock:
                self._is_connected = False
            self.message_handler.deferred_queue.put(
                cast(
                    ExecuteOnMainThreadFunction,
                    lambda context, discard, e=error: Client.raise_error(e),
                )
            )
        print(f"Client: Closed.")

    @staticmethod
    def raise_error(error: Exception) -> None:
        raise error
