import time
import traceback

import bpy
import bpy.ops
from bpy.types import Operator, Event, Object

from ..core.client import Client
from ..core.message_handler import (
    MessageHandler,
    UserFacingError,
    supported_rotation_modes,
)
from ..core.preset_storage import preset_from_property
from ..properties import Context, MappingItem, TPoseMappingItem

client: Client | None = None


class ClientStart(Operator):
    bl_idname = "mb.client_start"
    bl_label = "Connect"
    bl_description = "Start communicating data to Marionette"
    bl_options = {"REGISTER", "INTERNAL", "UNDO"}
    max_task_time_s: float = 1 / 30

    def modal(self, context: Context, event: Event) -> set[str]:  # type: ignore[override]
        global client

        if not context.scene.mb_wants_to_connect and client is not None:
            client.close()
            client = None

        if client is None or client.closed:
            return {"FINISHED"}

        # This gets run every frame
        try:
            if event.type == "TIMER":
                start_time = time.time()
                while not client.message_handler.deferred_queue.empty():
                    task = client.message_handler.deferred_queue.get_nowait()
                    task(context, time.time() - start_time > self.max_task_time_s)
                    client.message_handler.deferred_queue.task_done()

                while not client.message_handler.input_queue.empty():
                    task = client.message_handler.input_queue.get()
                    res = task(context, time.time() - start_time > self.max_task_time_s)
                    client.message_handler.output_queue.put(res)
                    client.message_handler.input_queue.task_done()
                bpy.context.view_layer.update()
        except UserFacingError as e:
            self.report({"ERROR"}, str(e))
            client.close()
            client = None
            return {"FINISHED"}
        except Exception:
            self.report({"ERROR"}, traceback.format_exc())
            client.close()
            client = None
            return {"FINISHED"}

        return {"PASS_THROUGH"}

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        global client

        if context.scene.mb_control_rig is None:
            self.report({"ERROR"}, "No rig selected")
            return {"CANCELLED"}

        if (
            context.scene.mb_selected_preset_index < 0
            or context.scene.mb_selected_preset_index >= len(context.scene.mb_presets)
        ):
            self.report({"ERROR"}, "No preset selected")
            return {"CANCELLED"}

        item: MappingItem
        for item in context.scene.mb_presets[
            context.scene.mb_selected_preset_index
        ].bone_map:
            error = item.error_message(context)
            if error is not None:
                self.report({"ERROR"}, error)
                return {"CANCELLED"}

        for item in context.scene.mb_presets[
            context.scene.mb_selected_preset_index
        ].bone_map:
            if item.bone_name == "":
                continue
            bone = context.scene.mb_control_rig.pose.bones[item.bone_name]
            if (
                item.copy_rotation
                and bone.rotation_mode not in supported_rotation_modes
            ):
                self.report(
                    {"WARNING"},
                    f"Unsupported rotation mode {bone.rotation_mode}, converting to Euler XYZ",
                )
                bone.rotation_mode = "XYZ"
            tpose_item: TPoseMappingItem
            for tpose_item in context.scene.mb_presets[
                context.scene.mb_selected_preset_index
            ].tpose:
                if tpose_item.bone_name == item.bone_name:
                    tpose_item.rotation_mode = bone.rotation_mode
                    break
        # TODO: VALIDATION
        # for item in selected_preset["bone_map"]:
        #     error = preset.mapping_error_message(
        #         item.bone_name,
        #         item.human_name,
        #         context.scene.mb_control_rig,
        #         selected_preset,
        #     )
        #     if error is not None:
        #         self.report({"ERROR"}, error)
        #         return {"CANCELLED"}

        # If animation is currently playing, stop it
        if context.screen.is_animation_playing:
            bpy.ops.screen.animation_cancel()

        # Start the client
        try:
            assert client is None
            client = Client(
                "127.0.0.1",
                11183,
                preset_from_property(
                    context.scene.mb_presets[context.scene.mb_selected_preset_index]
                ),
            )
        except OSError as e:
            print("Socket error:", e.strerror)
            self.report({"ERROR"}, e.strerror)
            return {"FINISHED"}

        # Register this classes modal operator in Blenders event handling system and execute it at the specified fps
        context.window_manager.modal_handler_add(self)
        context.window_manager.event_timer_add(
            1 / context.scene.mb_client_fps, window=bpy.context.window
        )
        context.scene.mb_wants_to_connect = True
        return {"RUNNING_MODAL"}

    @classmethod
    def poll(cls, context: Context) -> bool:  # type: ignore[override]
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        if context.scene.mb_control_rig is None:
            cls.poll_message_set("No rig selected")
            return False

        if (
            context.scene.mb_selected_preset_index < 0
            or context.scene.mb_selected_preset_index >= len(context.scene.mb_presets)
        ):
            cls.poll_message_set("No preset selected")
            return False
        if client is not None:
            return False
        return True


class ClientStop(Operator):
    bl_idname: str = "mb.client_stop"
    bl_label: str = "Disconnect"
    bl_description: str = "Stop receiving data from Marionette"
    bl_options: set[str] = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        context.scene.mb_wants_to_connect = False
        return {"FINISHED"}
