import json
from pathlib import Path
from typing import cast, TypedDict, Final, Any

import bpy
from bpy.app.handlers import persistent

from .. import mapping
from ..mapping import Mapping
from ..properties import (
    Scene,
    Position,
    Rotation,
    PresetItem,
    MappingItem,
    TPoseMappingItem,
)

# region V1
MappingV1 = tuple[str, str, bool, bool]
FileV1 = dict[str, list[MappingV1]]
# endregion


# region V2
MappingV2 = tuple[str, str, bool, bool]


class PresetV2(TypedDict):
    enable_ik: list[bool]
    bone_map: list[MappingV2]


FileV2 = dict[str, PresetV2]
# endregion


# region V3
MappingV3 = tuple[str, str, bool, bool]


class TPoseMappingV3(TypedDict):
    bone_name: str
    local_position: Position
    world_position: Position
    local_rotation: Rotation
    world_rotation: Rotation
    rotation_mode: str


class PresetV3(TypedDict):
    enable_ik: list[bool]
    bone_map: list[MappingV3]
    tpose: list[TPoseMappingV3]


class FileV3(TypedDict):
    version: int
    presets: dict[str, PresetV3]


# endregion

Preset = PresetV3

presets_file_path: Final = Path(__file__).parent.parent / "presets" / "preset.json"


def preset_from_property(p: PresetItem) -> Preset:
    return {
        "bone_map": [
            (b.human_name, b.bone_name, b.copy_rotation, b.copy_position)
            for b in p.bone_map
        ],
        "enable_ik": [
            p.left_arm_ik,
            p.right_arm_ik,
            p.left_leg_ik,
            p.right_leg_ik,
        ],
        "tpose": [
            cast(
                TPoseMappingV3,
                {
                    "bone_name": t.bone_name,
                    "local_position": (
                        t.local_position[0],
                        t.local_position[1],
                        t.local_position[2],
                    ),
                    "local_rotation": (
                        t.local_rotation[0],
                        t.local_rotation[1],
                        t.local_rotation[2],
                        t.local_rotation[3],
                    ),
                    "world_position": (
                        t.world_position[0],
                        t.world_position[1],
                        t.world_position[2],
                    ),
                    "world_rotation": (
                        t.world_rotation[0],
                        t.world_rotation[1],
                        t.world_rotation[2],
                        t.world_rotation[3],
                    ),
                    "rotation_mode": t.rotation_mode,
                },
            )
            for t in p.tpose
        ],
    }


def write_to_file() -> None:
    presets_file_path.parent.mkdir(parents=True, exist_ok=True)
    with open(presets_file_path, "w", encoding="utf8") as f:
        p: PresetItem
        json.dump(
            {
                "version": 3,
                "program": "blender",
                "presets": {
                    p.name: preset_from_property(p)
                    for p in cast(Scene, bpy.context.scene).mb_presets
                },
            },
            f,
        )

def write_preset_to_file(preset: PresetItem, path: Path) -> None:
    path.parent.mkdir(parents=True, exist_ok=True)
    with open(path, "w", encoding="utf8") as f:
        json.dump(
            {
                "version": 3,
                "program": "blender",
                "presets": { preset.name: preset_from_property(preset) },
            },
            f,
        )


@persistent  # type: ignore[misc]
def read_from_file() -> None:
    try:
        with open(presets_file_path, "r", encoding="utf8") as f:
            scene = cast(Scene, bpy.context.scene)
            scene.mb_presets.clear()
            for name, preset in upgrade_to_v3(json.load(f))["presets"].items():
                preset_item: PresetItem = scene.mb_presets.add()
                preset_item.name = name
                preset_item.bone_map.clear()
                for (
                    human_name,
                    bone_name,
                    copy_rotation,
                    copy_position,
                ) in preset["bone_map"]:
                    item = preset_item.bone_map.add()
                    item.human_name = human_name
                    item.bone_name = bone_name
                    item.copy_rotation = copy_rotation
                    item.copy_position = copy_position
                preset_item.tpose.clear()
                for mapping in preset["tpose"]:
                    item = preset_item.tpose.add()
                    item.bone_name = mapping["bone_name"]
                    item.local_position = mapping["local_position"]
                    item.local_rotation = mapping["local_rotation"]
                    item.world_position = mapping["world_position"]
                    item.world_rotation = mapping["world_rotation"]
                    item.rotation_mode = mapping["rotation_mode"]
                preset_item.left_arm_ik = preset["enable_ik"][0]
                preset_item.right_arm_ik = preset["enable_ik"][1]
                preset_item.left_leg_ik = preset["enable_ik"][2]
                preset_item.right_leg_ik = preset["enable_ik"][3]
    except FileNotFoundError:
        print("Preset file not found.")
    except json.decoder.JSONDecodeError:
        print("Preset file has invalid JSON.")


def append_from_file(path: Path) -> None:
    try:
        with open(path, "r", encoding="utf8") as f:
            scene = cast(Scene, bpy.context.scene)
            for name, preset in upgrade_to_v3(json.load(f))["presets"].items():
                renamed = name
                i = 0
                while renamed in scene.mb_presets:
                    i += 1
                    renamed = f"{name} ({i})"
                preset_item: PresetItem = scene.mb_presets.add()
                preset_item.name = renamed
                preset_item.bone_map.clear()
                for (
                    human_name,
                    bone_name,
                    copy_rotation,
                    copy_position,
                ) in preset["bone_map"]:
                    item = preset_item.bone_map.add()
                    item.human_name = human_name
                    item.bone_name = bone_name
                    item.copy_rotation = copy_rotation
                    item.copy_position = copy_position
                preset_item.tpose.clear()
                for mapping in preset["tpose"]:
                    item = preset_item.tpose.add()
                    item.bone_name = mapping["bone_name"]
                    item.local_position = mapping["local_position"]
                    item.local_rotation = mapping["local_rotation"]
                    item.world_position = mapping["world_position"]
                    item.world_rotation = mapping["world_rotation"]
                    item.rotation_mode = mapping["rotation_mode"]
                preset_item.left_arm_ik = preset["enable_ik"][0]
                preset_item.right_arm_ik = preset["enable_ik"][1]
                preset_item.left_leg_ik = preset["enable_ik"][2]
                preset_item.right_leg_ik = preset["enable_ik"][3]
    except FileNotFoundError:
        print("Preset file not found.")
    except json.decoder.JSONDecodeError:
        print("Preset file has invalid JSON.")


def upgrade_to_v3(json_data: dict[str, Any]) -> FileV3:
    if "version" in json_data and json_data["version"] == 3:
        return cast(FileV3, json_data)

    def upgrade(old: PresetV2) -> PresetV3:
        return {"enable_ik": old["enable_ik"], "bone_map": old["bone_map"], "tpose": []}

    return {
        "version": 3,
        "program": "blender",
        "presets": {k: upgrade(p) for k, p in upgrade_to_v2(json_data).items()},
    }


def upgrade_to_v2(
    data: dict[str, list[Mapping] | PresetV2],
) -> FileV2:
    upgraded: FileV2 = {}
    for key in data:
        if isinstance(data[key], list):
            bone_map = cast(list[Mapping], data[key])
            enable_ik = [False, False, False, False]
            for i, (human_name, _, _, _) in enumerate(mapping.get_default_map()):
                is_found = False
                for mapped_human_name, bone_name, _, _ in bone_map:
                    if mapped_human_name == human_name:
                        is_found = True
                        if human_name == "LeftArmBendGoal" and bone_name != "":
                            enable_ik[0] = True
                        elif human_name == "RightArmBendGoal" and bone_name != "":
                            enable_ik[1] = True
                        elif human_name == "LeftLegBendGoal" and bone_name != "":
                            enable_ik[2] = True
                        elif human_name == "RightLegBendGoal" and bone_name != "":
                            enable_ik[3] = True
                        break
                if not is_found:
                    bone_map.insert(i, (human_name, "", True, True))
        else:
            preset_dict = cast(PresetV2, data[key])
            bone_map = preset_dict["bone_map"]
            enable_ik = preset_dict["enable_ik"]
        upgraded[key] = {"bone_map": bone_map, "enable_ik": enable_ik}
    return upgraded
