from pathlib import Path
import re

from statistics import quantiles

import bpy
from bpy_extras.io_utils import ImportHelper, ExportHelper
from bpy.props import StringProperty, EnumProperty
from bpy.types import PoseBone, Object
from mathutils import Quaternion, Matrix

from . import client_operators
from ..core import preset_storage
from .. import mapping
from ..core.message_handler import MessageHandler
from ..properties import (
    Context,
    Scene,
    PresetItem,
    MappingItem,
    TPoseMappingItem,
    bpy_prop_collection_idprop,
)


def get_preset_names(self: Scene, context: Context) -> list[tuple[str, str, str, int]]:
    # (identifier, name, description, icon, number)
    preset_enum: list[tuple[str, str, str, int]] = []
    for p in context.scene.mb_presets:
        preset_enum.append((str(p.name), str(p.name), "", (len(preset_enum))))
    return preset_enum


def copy_pose_to(
    rig: Object, pose: bpy_prop_collection_idprop[TPoseMappingItem]
) -> None:
    pose.clear()
    for bone in rig.pose.bones:
        bone_pose: TPoseMappingItem = pose.add()
        bone_pose.bone_name = bone.name
        bone_pose.local_position = bone.location
        if bone.rotation_mode == "QUATERNION":
            bone_pose.local_rotation = bone.rotation_quaternion
        elif bone.rotation_mode == "AXIS_ANGLE":
            angle, *axis = bone.rotation_axis_angle
            bone_pose.local_rotation = Matrix.Rotation(angle, 4, axis).to_quaternion()
        else:
            bone_pose.local_rotation = bone.rotation_euler.to_quaternion()
        world_pos, world_rot = MessageHandler.get_world_coords(
            rig,
            bone,
        )
        bone_pose.world_position = world_pos
        bone_pose.world_rotation = world_rot
        bone_pose.rotation_mode = bone.rotation_mode


def try_sample_pose(
    rig: Object, pose: bpy_prop_collection_idprop[TPoseMappingItem]
) -> str | None:
    bone_poses = {p.bone_name: p for p in pose}
    bones = {b.name: b for b in rig.pose.bones}
    if len(bone_poses) != len(bones):
        return "Unable to restore pose: Rig does not match"
    for bone_name, bone_pose in bone_poses.items():
        if bone_name not in bones:
            return "Unable to restore pose: Rig does not match"
        bone = bones[bone_name]
        if bone.rotation_mode != bone_pose.rotation_mode:
            return "Unable to restore pose: Rig does not match"
    for bone_name, bone_pose in bone_poses.items():
        bone = bones[bone_name]
        bone.location = bone_pose.local_position
        if bone.rotation_mode == "QUATERNION":
            bone.rotation_quaternion = bone_pose.local_rotation
        elif bone.rotation_mode == "AXIS_ANGLE":
            axis, angle = Quaternion(bone_pose.local_rotation).to_axis_angle()
            bone.rotation_axis_angle = [angle, *axis]
        else:
            bone.rotation_euler = Quaternion(bone_pose.local_rotation).to_euler(
                bone.rotation_mode
            )  # type: ignore[call-arg]
    return None


class PresetAdd(bpy.types.Operator):
    bl_idname: str = "mb.preset_add"
    bl_label: str = "Add Preset"
    bl_description: str = "Add a new preset for rig mapping"
    bl_options: set[str] = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        if not context.scene.mb_control_rig:
            self.report({"ERROR"}, "Rig does not exist")
            return {"CANCELLED"}
        if client_operators.client is not None:
            client_operators.client.close()
            client_operators.client = None
        preset: PresetItem = context.scene.mb_presets.add()
        name = "Preset"
        i = 1
        p: PresetItem
        while name in [p.name for p in context.scene.mb_presets]:
            name = f"Preset.{i:03d}"
            i += 1
        preset.name = name
        preset.bone_map.clear()
        for (
            human_name,
            bone_name,
            copy_rotation,
            copy_position,
        ) in mapping.get_default_map():
            item: MappingItem = preset.bone_map.add()
            item.human_name = human_name
            item.bone_name = bone_name
            item.copy_rotation = copy_rotation
            item.copy_position = copy_position
        copy_pose_to(context.scene.mb_control_rig, context.scene.mb_pose_before_editing)
        context.scene.mb_editing_preset.copy_from(preset)
        context.scene.mb_editing_preset_old_name = preset.name
        context.scene.mb_editing_preset_is_new = True
        preset_storage.write_to_file()

        return {"FINISHED"}

    @classmethod
    def poll(cls, context: Context) -> bool:  # type: ignore[override]
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        if context.scene.mb_editing_preset_old_name != "":
            cls.poll_message_set("Can't add a new preset while editing a preset")
            return False
        return True


class PresetRemove(bpy.types.Operator):
    bl_idname: str = "mb.preset_remove"
    bl_label: str = "Remove Preset"
    bl_description: str = "Removes the selected rig mapping preset"
    bl_options: set[str] = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        if client_operators.client is not None:
            client_operators.client.close()
            client_operators.client = None
        context.scene.mb_presets.remove(context.scene.mb_selected_preset_index)
        if context.scene.mb_selected_preset_index >= len(context.scene.mb_presets):
            context.scene.mb_selected_preset_index = len(context.scene.mb_presets) - 1
        preset_storage.write_to_file()
        return {"FINISHED"}

    @classmethod
    def poll(cls, context: Context) -> bool:  # type: ignore[override]
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        if context.scene.mb_editing_preset_old_name != "":
            cls.poll_message_set("Can't remove a new preset while editing a preset")
            return False
        return True


class PresetImport(bpy.types.Operator, ImportHelper):
    bl_idname: str = "mb.preset_import"
    bl_label: str = "Import Preset"
    bl_description: str = "Imports a preset from a file"
    bl_options: set[str] = {"REGISTER", "UNDO", "INTERNAL"}

    filter_glob: bpy.props.StringProperty(
        default="*.json",
        options={"HIDDEN"}
    )

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        preset_storage.append_from_file(Path(self.filepath))
        return {"FINISHED"}

    @classmethod
    def poll(cls, context: Context) -> bool:  # type: ignore[override]
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        if context.scene.mb_editing_preset_old_name != "":
            cls.poll_message_set("Can't import a new preset while editing a preset")
            return False
        return True


class PresetExport(bpy.types.Operator, ExportHelper):
    bl_idname: str = "mb.preset_export"
    bl_label: str = "Export Preset"
    bl_description: str = "Exports a preset to a file"
    bl_options: set[str] = {"REGISTER", "INTERNAL"}

    preset_name: EnumProperty(name="Preset Name", items=get_preset_names, options={"HIDDEN"})  # type: ignore[valid-type]

    filename_ext: str = ".json"
    filter_glob: bpy.props.StringProperty(
        default="*.json",
        options={"HIDDEN"}
    )

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        for i, p in enumerate(context.scene.mb_presets):
            if p.name == self.preset_name:
                preset_storage.write_preset_to_file(p, Path(self.filepath))
                return {"FINISHED"}

        self.report({"ERROR"}, "Failed to find preset")
        return {"CANCELLED"}

    def invoke(self, context, event):
        filename = self.preset_name
        filename = re.sub(r"[<>:\"/\\|?*\x00-\x1f]", "", filename)
        filename = filename.rstrip('. ')
        if not filename:
            filename = "Marioentte Preset"
        self.filepath = filename + self.filename_ext
        context.window_manager.fileselect_add(self)
        return {'RUNNING_MODAL'}


class PresetMove(bpy.types.Operator):
    bl_idname: str = "mb.preset_move"
    bl_label: str = "Move Preset"
    bl_description: str = "Moves the selected rig mapping preset"
    bl_options: set[str] = {"REGISTER", "UNDO", "INTERNAL"}

    direction: EnumProperty(  # type: ignore[valid-type]
        name="Direction",
        items=[
            ("UP", "Up", "Move one slot up"),
            ("DOWN", "Down", "Move one slot down"),
        ],
    )

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        if client_operators.client is not None:
            client_operators.client.close()
            client_operators.client = None
        _from = context.scene.mb_selected_preset_index
        to = (
            max(_from - 1, 0)
            if self.direction == "UP"
            else min(_from + 1, len(context.scene.mb_presets) - 1)
        )
        context.scene.mb_presets.move(_from, to)
        context.scene.mb_selected_preset_index = to
        preset_storage.write_to_file()
        return {"FINISHED"}

    @classmethod
    def poll(cls, context: Context) -> bool:  # type: ignore[override]
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        if context.scene.mb_editing_preset_old_name != "":
            cls.poll_message_set("Can't move a preset while editing a preset")
            return False
        return True


class EditPreset(bpy.types.Operator):
    bl_idname: str = "mb.edit_preset"
    bl_label: str = "Edit Preset"
    bl_description: str = "Edit the bone mappings and t-pose of the preset"
    bl_options: set[str] = {"REGISTER", "UNDO", "INTERNAL"}

    preset_name: EnumProperty(name="Preset Name", items=get_preset_names)  # type: ignore[valid-type]

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        if not context.scene.mb_control_rig:
            context.scene.mb_editing_preset_old_name = ""
            self.report({"ERROR"}, "Rig does not exist")
            return {"CANCELLED"}
        if client_operators.client is not None:
            client_operators.client.close()
            client_operators.client = None
        for o in bpy.context.selected_objects:
            o.select_set(False)
        if bpy.context.active_object and bpy.context.active_object.mode == "POSE":
            bpy.ops.object.posemode_toggle()
        context.scene.mb_control_rig.select_set(True)
        bpy.context.view_layer.objects.active = context.scene.mb_control_rig
        if context.scene.mb_control_rig.mode != "POSE":
            bpy.ops.object.posemode_toggle()
        copy_pose_to(context.scene.mb_control_rig, context.scene.mb_pose_before_editing)

        for i, p in enumerate(context.scene.mb_presets):
            if p.name == self.preset_name:
                context.scene.mb_editing_preset.copy_from(p)
                context.scene.mb_editing_preset_old_name = p.name
                context.scene.mb_editing_preset_is_new = False
                error = try_sample_pose(
                    context.scene.mb_control_rig, context.scene.mb_editing_preset.tpose
                )
                if error is not None:
                    self.report({"WARNING"}, error)
                    context.scene.mb_editing_preset_message = (
                        "Saved T-Pose was not compatible with this rig."
                    )
                else:
                    context.scene.mb_editing_preset_message = ""
                return {"FINISHED"}

        self.report({"ERROR"}, "Failed to find preset")
        return {"CANCELLED"}

    @classmethod
    def poll(cls, context: Context) -> bool:  # type: ignore[override]
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        if context.scene.mb_editing_preset_old_name != "":
            cls.poll_message_set(
                "Can't start editing a preset while already editing a preset"
            )
            return False
        return True


class EditPresetCancel(bpy.types.Operator):
    bl_idname: str = "mb.edit_preset_cancel"
    bl_label: str = "Discard Changes"
    bl_description: str = "Discard any changes made to the preset"
    bl_options: set[str] = {"REGISTER", "UNDO", "INTERNAL"}

    name: StringProperty(default="Unnamed preset", name="Preset name")  # type: ignore[valid-type]

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        context.scene.mb_editing_preset_message = ""
        if client_operators.client is not None:
            client_operators.client.close()
            client_operators.client = None
        if context.scene.mb_editing_preset_is_new:
            found_index = -1
            for i, p in enumerate(context.scene.mb_presets):
                if p.name == context.scene.mb_editing_preset_old_name:
                    found_index = i
                    break
            if found_index != -1:
                context.scene.mb_presets.remove(found_index)
                preset_storage.write_to_file()
        context.scene.mb_editing_preset_old_name = ""
        if context.scene.mb_control_rig:
            error = try_sample_pose(
                context.scene.mb_control_rig, context.scene.mb_pose_before_editing
            )
            if error is not None:
                self.report({"WARNING"}, error)
        return {"FINISHED"}

    @classmethod
    def poll(cls, context: Context) -> bool:  # type: ignore[override]
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        if context.scene.mb_editing_preset_old_name == "":
            cls.poll_message_set("Not editing any preset right now")
            return False
        return True


class EditPresetSave(bpy.types.Operator):
    bl_idname: str = "mb.edit_preset_save"
    bl_label: str = "Save"
    bl_description: str = "Save changes to the preset"
    bl_options: set[str] = {"REGISTER", "UNDO", "INTERNAL"}

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        context.scene.mb_editing_preset_message = ""
        if client_operators.client is not None:
            client_operators.client.close()
            client_operators.client = None
        if not context.scene.mb_control_rig:
            context.scene.mb_editing_preset_old_name = ""
            self.report({"ERROR"}, "Rig does not exist")
            return {"CANCELLED"}
        copy_pose_to(
            context.scene.mb_control_rig, context.scene.mb_editing_preset.tpose
        )

        found_index = -1
        for i, p in enumerate(context.scene.mb_presets):
            if p.name == context.scene.mb_editing_preset_old_name:
                found_index = i
                break
        if found_index != -1:
            context.scene.mb_presets[found_index].copy_from(
                context.scene.mb_editing_preset
            )
            context.scene.mb_selected_preset_index = found_index
        else:
            context.scene.mb_presets.add().copy_from(context.scene.mb_editing_preset)
            context.scene.mb_selected_preset_index = len(context.scene.mb_presets) - 1
        preset_storage.write_to_file()
        context.scene.mb_editing_preset_old_name = ""
        error = try_sample_pose(
            context.scene.mb_control_rig, context.scene.mb_pose_before_editing
        )
        if error is not None:
            self.report({"WARNING"}, error)
        return {"FINISHED"}

    # def invoke(self, context: Context, event: Event) -> Any:  # type: ignore[override]
    #     return context.window_manager.invoke_confirm(self, event)

    @classmethod
    def poll(cls, context: Context) -> bool:  # type: ignore[override]
        """Test if the operator can be called or not

        :param context:
        :type context: Context
        :return:
        :rtype: bool
        """
        if context.scene.mb_editing_preset_old_name == "":
            cls.poll_message_set("Can't save a preset if not in edit mode")
            return False
        if context.scene.mb_editing_preset.name == "":
            cls.poll_message_set("Preset name is required")
            return False
        return True


twin_labels = {
    "RightHand": "LeftHand",
    "RightLowerArm": "LeftLowerArm",
    "RightUpperArm": "LeftUpperArm",
    "RightShoulder": "LeftShoulder",
    "RightToes": "LeftToes",
    "RightFoot": "LeftFoot",
    "RightLowerLeg": "LeftLowerLeg",
    "RightUpperLeg": "LeftUpperLeg",
    "RightEye": "LeftEye",
    "RightThumbDistal": "LeftThumbDistal",
    "RightThumbIntermediate": "LeftThumbIntermediate",
    "RightThumbProximal": "LeftThumbProximal",
    "RightIndexDistal": "LeftIndexDistal",
    "RightIndexIntermediate": "LeftIndexIntermediate",
    "RightIndexProximal": "LeftIndexProximal",
    "RightMiddleDistal": "LeftMiddleDistal",
    "RightMiddleIntermediate": "LeftMiddleIntermediate",
    "RightMiddleProximal": "LeftMiddleProximal",
    "RightRingDistal": "LeftRingDistal",
    "RightRingIntermediate": "LeftRingIntermediate",
    "RightRingProximal": "LeftRingProximal",
    "RightLittleDistal": "LeftLittleDistal",
    "RightLittleIntermediate": "LeftLittleIntermediate",
    "RightLittleProximal": "LeftLittleProximal",
    "RightArmBendGoal": "LeftArmBendGoal",
    "RightLegBendGoal": "LeftLegBendGoal",
}
twin_labels = {**twin_labels, **{v: k for k, v in twin_labels.items()}}


class PickBone(bpy.types.Operator):
    bl_idname = "mb.pick_bone"
    bl_label = "Pick Bone"
    bl_description = "Maps the currently selected bone for Marionette retargeting."
    bl_options = {"REGISTER", "UNDO", "INTERNAL"}

    human_name: bpy.props.StringProperty(options={"HIDDEN"})  # type: ignore[valid-type]

    def execute(self, context: Context) -> set[str]:  # type: ignore[override]
        if context.object.mode != "POSE":
            self.report({"ERROR"}, "Not in pose mode")
            return {"CANCELLED"}
        if context.object.type != "ARMATURE":
            self.report({"ERROR"}, "Selected object is not an armature")
            return {"CANCELLED"}

        selected_pose_bone: PoseBone | None = None
        if len(context.selected_pose_bones_from_active_object) > 0:
            selected_pose_bone = context.selected_pose_bones_from_active_object[0]
        if selected_pose_bone is None:
            self.report({"ERROR"}, "No bone selected")
            return {"CANCELLED"}

        self.set_mapping(context, self.human_name, selected_pose_bone.name)
        if self.human_name in twin_labels and context.scene.mb_mirror_selection:
            twin = self.get_twin_name(selected_pose_bone.name)
            if twin is not None and twin in context.object.pose.bones:
                self.set_mapping(context, twin_labels[self.human_name], twin)

        return {"FINISHED"}

    @staticmethod
    def set_mapping(context: Context, human_name: str, bone_name: str) -> None:
        item: MappingItem
        for item in context.scene.mb_editing_preset.bone_map:
            if item.human_name == human_name:
                item.bone_name = bone_name
                break

    @staticmethod
    def get_twin_name(name: str) -> str | None:
        if name is None or name.strip() == "":
            return ""

        replacement_rules = [
            # Left, Right, Rule (contains, suffix, prefix)
            ["Left", "Right", "contains"],
            ["left", "right", "contains"],
            ["L", "R", "startswith"],
            ["L_", "R_", "startswith"],
            ["_L", "_R", "endswith"],
            ["lt_", "rt_", "startswith"],
            ["_L_", "_R_", "contains"],
            [".l", ".r", "endswith"],
            [".L", ".R", "endswith"],
        ]
        for left, right, rule in replacement_rules:
            for target, replacement in [[left, right], [right, left]]:
                if rule == "contains" and target in name:
                    return name.replace(target, replacement)
                elif rule == "startswith" and name.startswith(target):
                    return replacement + name[len(target) :]
                elif rule == "endswith" and name.endswith(target):
                    return name[: len(name) - len(target)] + replacement
        return None
