import re
from typing import Any, cast

from bpy.types import Panel, UIList, UILayout, UI_UL_list

from . import mapping
from .core.icon_manager import Icons
from .operators import client_operators, preset_operators
from .operators.open_website import OpenWebsite
from .properties import Context, PresetItem, MappingItem


# Initializes the Marionette panel in the toolbar
class ToolPanel:
    bl_label: str = "Marionette"
    bl_idname: str = "VIEW3D_TS_Marionette"
    bl_category: str = "Marionette"
    bl_space_type: str = "VIEW_3D"
    bl_region_type: str = "UI"


class InfoPanel(ToolPanel, Panel):
    bl_idname: str = "VIEW3D_PT_mb_info"
    bl_label: str = "Info"

    def draw(self, context: Context) -> None:  # type: ignore[override]
        layout = self.layout
        row = layout.row()
        row.label(text="Developed by Marionette ApS")
        row.operator(OpenWebsite.bl_idname, icon="URL")


class MarionettePanel(ToolPanel, Panel):
    bl_idname: str = "VIEW3D_PT_mb_marionette"
    bl_label: str = "Marionette"

    def draw(self, context: Context) -> None:  # type: ignore[override]
        if context.scene.mb_editing_preset_old_name != "":
            self.draw_edit_preset_screen(context)
        else:
            self.draw_primary_screen(context)

    def draw_primary_screen(self, context: Context) -> None:
        # self.layout.enabled = client_operators.client is None or (
        #     not client_operators.client.is_connected
        #     and not client_operators.client.wants_to_connect
        # )
        self.layout.use_property_split = False
        self.layout.label(
            text="Marionette for Blender", icon_value=Icons.LOGO.get_icon()
        )
        self.layout.prop(context.scene, "mb_control_rig", icon="OUTLINER_OB_ARMATURE")

        if not context.scene.mb_control_rig:
            return

        row = self.layout.row()
        row.template_list(
            MB_UL_presets.__name__,
            "presets",
            context.scene,
            "mb_presets",
            context.scene,
            "mb_selected_preset_index",
            rows=1,
            maxrows=10,
        )
        col = row.column(align=True)
        col.operator("mb.preset_add", icon="ADD", text="")
        col.operator("mb.preset_remove", icon="REMOVE", text="")
        col.separator()
        col.operator("mb.preset_import", icon="IMPORT", text="")
        # col.menu("DATA_MT_bone_group_context_menu", icon="DOWNARROW_HLT", text="")
        if context.scene.mb_presets:
            col.separator()
            col.operator("mb.preset_move", icon="TRIA_UP", text="").direction = "UP"  # type: ignore[attr-defined]
            col.operator("mb.preset_move", icon="TRIA_DOWN", text="").direction = "DOWN"  # type: ignore[attr-defined]

        if context.scene.mb_selected_preset_index == -1:
            return

        layout = self.layout
        layout.use_property_split = False
        col = layout.column()
        row = col.row()
        if client_operators.client is not None and client_operators.client.is_connected:
            row.label(text="Connected.")
        elif (
            client_operators.client is not None
            and not client_operators.client.is_connected
        ):
            row.label(text="Connecting...")
        elif client_operators.client is None:
            row.label(text="Disconnected.")
        if client_operators.client is not None:
            row.operator(
                client_operators.ClientStop.bl_idname, icon="PAUSE", depress=True
            )
        else:
            row.operator(client_operators.ClientStart.bl_idname, icon="PLAY")

    def draw_edit_preset_screen(self, context: Context) -> None:
        if context.scene.mb_editing_preset_message != "":
            self.layout.label(
                text=context.scene.mb_editing_preset_message, icon="ERROR"
            )

        self.layout.prop(context.scene.mb_editing_preset, "name", toggle=True)

        row = self.layout.row(align=True, heading="IK:")
        row.prop(context.scene.mb_editing_preset, "left_leg_ik", toggle=True)
        row.prop(context.scene.mb_editing_preset, "right_leg_ik", toggle=True)
        row.prop(context.scene.mb_editing_preset, "left_arm_ik", toggle=True)
        row.prop(context.scene.mb_editing_preset, "right_arm_ik", toggle=True)

        col = self.layout.column(align=True)
        col.template_list(
            MB_UL_preset_bone_map.__name__,
            "preset_bone_map",
            context.scene.mb_editing_preset,
            "bone_map",
            context.scene,
            "mb_editing_preset_bone_map_index",
            rows=1,
            maxrows=10,
        )

        split = self.layout.split()
        row = split.row()
        row.alignment = "LEFT"
        row.label(text="* required bone")
        row = split.row()
        row.alignment = "RIGHT"
        row.prop(context.scene, "mb_mirror_selection")

        row = self.layout.row(align=True)
        row.operator(
            preset_operators.EditPresetSave.bl_idname,
            icon="IMPORT",
        )
        row.separator()
        row.operator(
            preset_operators.EditPresetCancel.bl_idname,
            icon="CANCEL",
        )


class MB_UL_presets(UIList):
    def draw_item(
        self,
        context: Context,  # type: ignore[override]
        layout: UILayout,
        data: Any,
        item: PresetItem,  # type: ignore[override]
        icon: Any | None,
        active_data: int | None,
        active_property: str | None,
        index: Any | None = 0,
        flt_flag: Any | None = 0,
    ) -> None:
        layout.prop(item, "name", text="", emboss=False, icon="ARMATURE_DATA")
        layout.operator(
            preset_operators.EditPreset.bl_idname,
            text="",
            icon="GREASEPENCIL",
        ).preset_name = item.name  # type: ignore[attr-defined]
        layout.operator(
            preset_operators.PresetExport.bl_idname,
            text="",
            icon="EXPORT",
        ).preset_name = item.name  # type: ignore[attr-defined]


class MB_UL_preset_bone_map(UIList):
    def draw_item(
        self,
        context: Context,  # type: ignore[override]
        layout: UILayout,
        data: Any,
        item: MappingItem,  # type: ignore[override]
        icon: Any | None,
        active_data: int | None,
        active_property: str | None,
        index: Any | None = 0,
        flt_flag: Any | None = 0,
    ) -> None:
        layout.enabled = (
            (
                item.human_name != "LeftLegBendGoal"
                or context.scene.mb_editing_preset.left_leg_ik
            )
            and (
                item.human_name != "RightLegBendGoal"
                or context.scene.mb_editing_preset.right_leg_ik
            )
            and (
                item.human_name != "LeftArmBendGoal"
                or context.scene.mb_editing_preset.left_arm_ik
            )
            and (
                item.human_name != "RightArmBendGoal"
                or context.scene.mb_editing_preset.right_arm_ik
            )
        )
        row = layout.row()

        # Displays marionette bone name
        display_name = self.humanize(item.human_name)
        if item.human_name in mapping.required_bones:
            display_name += " *"
        row.label(text=display_name)

        # Displays target bone
        sub = row.row(align=True)
        if context.scene.mb_control_rig:
            sub.prop_search(
                item,
                "bone_name",
                context.scene.mb_control_rig.pose,
                "bones",
                icon=item.icon(context),
            )
            picker = cast(
                preset_operators.PickBone,
                sub.operator(
                    preset_operators.PickBone.bl_idname, text="", icon="EYEDROPPER"
                ),
            )
            picker.human_name = item.human_name

            sub = row.row(align=True)
            sub.prop(item, "copy_rotation", text="", icon="CON_ROTLIKE", toggle=True)
            sub.prop(item, "copy_position", text="", icon="CON_LOCLIKE", toggle=True)

    def filter_items(
        self, context: Context, data: PresetItem, property: str  # type: ignore[override]
    ) -> tuple[list[int], list[int]]:
        # This function gets the collection property (as the usual tuple (data, propname)), and must return two lists:
        # * The first one is for filtering, it must contain 32bit integers were self.bitflag_filter_item marks the
        #   matching item as filtered (i.e. to be shown), and 31 other bits are free for custom needs.
        # * The second one is for reordering, it must return a list containing the new indices of the items (which
        #   gives us a mapping org_idx -> new_idx).
        # Please note that the default UI_UL_list defines helper functions for common tasks (see its doc for more info).
        # If you do not make filtering and/or ordering, return empty list(s) (this will be more efficient than
        # returning full lists doing nothing!).
        all_items = getattr(data, property)

        shown: list[int] = []
        if self.filter_name:
            shown = [
                (
                    self.bitflag_filter_item
                    if self.filter_name.lower() in self.humanize(i.human_name).lower()
                    else 0
                )
                for i in all_items
            ]

        order: list[int] = []
        if self.use_filter_sort_alpha:
            order = UI_UL_list.sort_items_by_name(all_items, "human_name")  # type: ignore[attr-defined]

        return shown, order

    @staticmethod
    def humanize(name: str) -> str:
        return re.sub(r"(?<=[a-z])(?=[A-Z])", " ", name)
