import os
from typing import Optional

import maya.cmds as cmds

from marionette.mapping import required_bones, mapping_order
from marionette.preset_storage import PresetStorage, Preset

twin_labels = {
    'RightHand': 'LeftHand',
    'RightLowerArm': 'LeftLowerArm',
    'RightUpperArm': 'LeftUpperArm',
    'RightShoulder': 'LeftShoulder',
    'RightToes': 'LeftToes',
    'RightFoot': 'LeftFoot',
    'RightLowerLeg': 'LeftLowerLeg',
    'RightUpperLeg': 'LeftUpperLeg',
    'RightEye': 'LeftEye',
    'RightThumbDistal': 'LeftThumbDistal',
    'RightThumbIntermediate': 'LeftThumbIntermediate',
    'RightThumbProximal': 'LeftThumbProximal',
    'RightIndexDistal': 'LeftIndexDistal',
    'RightIndexIntermediate': 'LeftIndexIntermediate',
    'RightIndexProximal': 'LeftIndexProximal',
    'RightMiddleDistal': 'LeftMiddleDistal',
    'RightMiddleIntermediate': 'LeftMiddleIntermediate',
    'RightMiddleProximal': 'LeftMiddleProximal',
    'RightRingDistal': 'LeftRingDistal',
    'RightRingIntermediate': 'LeftRingIntermediate',
    'RightRingProximal': 'LeftRingProximal',
    'RightLittleDistal': 'LeftLittleDistal',
    'RightLittleIntermediate': 'LeftLittleIntermediate',
    'RightLittleProximal': 'LeftLittleProximal',
    'RightArmBendGoal': 'LeftArmBendGoal',
    'RightLegBendGoal': 'LeftLegBendGoal',
}
twin_labels = {**twin_labels, **{v: k for k, v in twin_labels.items()}}


def get_twin_name(name: str) -> Optional[str]:
    if name is None or name.strip() == '':
        return ''

    replacement_rules = [
        # Left, Right, Rule (contains, suffix, prefix)
        ["Left", "Right", "contains"],
        ["left", "right", "contains"],
        ["L", "R", "startswith"],
        ["L_", "R_", "startswith"],
        ["_L", "_R", "endswith"],
        ["lt_", "rt_", "startswith"],
        ["_L_", "_R_", "contains"],
        [".l", ".r", "endswith"],
        [".L", ".R", "endswith"],
    ]
    for left, right, rule in replacement_rules:
        for target, replacement in [[left, right], [right, left]]:
            if rule == 'contains' and target in name:
                return name.replace(target, replacement)
            elif rule == 'startswith' and name.startswith(target):
                return replacement + name[len(target):]
            elif rule == 'endswith' and name.endswith(target):
                return name[:len(name) - len(target)] + replacement
    return None


image_paths = [os.path.join(os.path.dirname(__file__), 'RightHand.png'),
               os.path.join(os.path.dirname(__file__), 'Character.png'),
               os.path.join(os.path.dirname(__file__), 'LeftHand.png')]

required_empty_icon_name = os.path.join(os.path.dirname(__file__), 'AddRequired.png')  # 'menuIconAdd.png'
optional_empty_icon_name = os.path.join(os.path.dirname(__file__), 'AddOptional.png')  # 'item_add.png'
filled_ok_icon_name = 'confirm.png'
filled_error_icon_name = 'error.png'

# Info list (label, x, y)
info_list = [('RightHand', 215, 290),
             ('RightLowerArm', 220, 230),
             ('RightArmBendGoal', 220 - 50, 230),
             ('RightUpperArm', 225, 170),
             ('RightShoulder', 250, 135),

             ('LeftHand', 390, 290),
             ('LeftLowerArm', 385, 230),
             ('LeftArmBendGoal', 385 + 50, 230),
             ('LeftUpperArm', 380, 170),
             ('LeftShoulder', 360, 135),

             ('RightToes', 255, 470),
             ('RightFoot', 270, 440),
             ('RightLowerLeg', 270, 380),
             ('RightLegBendGoal', 270 - 50, 380),
             ('RightUpperLeg', 270, 325),

             ('LeftToes', 350, 470),
             ('LeftFoot', 335, 440),
             ('LeftLowerLeg', 335, 380),
             ('LeftLegBendGoal', 335 + 50, 380),
             ('LeftUpperLeg', 335, 325),

             ('Hips', 305, 285),
             ('Spine', 305, 240),
             ('Chest', 305, 195),
             ('UpperChest', 305, 150),
             ('Neck', 305, 110),
             ('Head', 305, 70),

             ('RightEye', 275, 60),
             ('LeftEye', 335, 60),

             ('RightThumbDistal', 125, 245),
             ('RightThumbIntermediate', 110, 270),
             ('RightThumbProximal', 95, 295),

             ('RightIndexDistal', 95, 190),
             ('RightIndexIntermediate', 90, 215),
             ('RightIndexProximal', 85, 240),

             ('RightMiddleDistal', 70, 185),
             ('RightMiddleIntermediate', 69, 210),
             ('RightMiddleProximal', 68, 235),

             ('RightRingDistal', 45, 190),
             ('RightRingIntermediate', 47, 215),
             ('RightRingProximal', 50, 240),

             ('RightLittleDistal', 20, 215),
             ('RightLittleIntermediate', 27, 240),
             ('RightLittleProximal', 35, 265),

             ('LeftThumbDistal', 475, 245),
             ('LeftThumbIntermediate', 490, 270),
             ('LeftThumbProximal', 505, 295),

             ('LeftIndexDistal', 510, 190),
             ('LeftIndexIntermediate', 515, 215),
             ('LeftIndexProximal', 520, 240),

             ('LeftMiddleDistal', 535, 185),
             ('LeftMiddleIntermediate', 536, 210),
             ('LeftMiddleProximal', 538, 235),

             ('LeftRingDistal', 560, 190),
             ('LeftRingIntermediate', 557, 215),
             ('LeftRingProximal', 555, 240),

             ('LeftLittleDistal', 585, 215),
             ('LeftLittleIntermediate', 577, 240),
             ('LeftLittleProximal', 570, 265)]

ik_list = [('LeftArmBendGoal', 385 + 50 + 4, 230 + 60),
           ('RightArmBendGoal', 220 - 50 - 4, 230 + 60),
           ('LeftLegBendGoal', 335 + 50 + 4, 380 + 60),
           ('RightLegBendGoal', 270 - 50 - 4, 380 + 60), ]

body_bones = ['Hips', 'Spine', 'Chest', 'UpperChest',
              'LeftShoulder', 'LeftUpperArm', 'LeftLowerArm', 'LeftArmBendGoal', 'LeftHand',
              'RightShoulder', 'RightUpperArm', 'RightLowerArm', 'RightArmBendGoal', 'RightHand',
              'LeftUpperLeg', 'LeftLowerLeg', 'LeftLegBendGoal', 'LeftFoot', 'LeftToes',
              'RightUpperLeg', 'RightLowerLeg', 'RightLegBendGoal', 'RightFoot', 'RightToes', ]
head_bones = ['Neck', 'Head', 'LeftEye', 'RightEye', 'Jaw']
left_hand_bones = ['LeftThumbDistal', 'LeftThumbIntermediate', 'LeftThumbProximal', 'LeftIndexDistal',
                   'LeftIndexIntermediate', 'LeftIndexProximal', 'LeftMiddleDistal', 'LeftMiddleIntermediate',
                   'LeftMiddleProximal', 'LeftRingDistal', 'LeftRingIntermediate', 'LeftRingProximal',
                   'LeftLittleDistal', 'LeftLittleIntermediate', 'LeftLittleProximal', ]
right_hand_bones = ['RightThumbDistal', 'RightThumbIntermediate', 'RightThumbProximal', 'RightIndexDistal',
                    'RightIndexIntermediate', 'RightIndexProximal', 'RightMiddleDistal', 'RightMiddleIntermediate',
                    'RightMiddleProximal', 'RightRingDistal', 'RightRingIntermediate', 'RightRingProximal',
                    'RightLittleDistal', 'RightLittleIntermediate', 'RightLittleProximal', ]
tab_bones = [body_bones, head_bones, left_hand_bones, right_hand_bones]

WINDOW_NAME = '__marionette__marionette_maya_plugin_main_window'


class MarionetteWindow:
    def __init__(self) -> None:
        self.preset_storage: PresetStorage = PresetStorage()
        self.selected_preset: Optional[int] = None
        self.selected_preset_is_changing: bool = False
        self._controller_text_fields: dict[str, str] = {}
        self._controller_buttons: dict[str, str] = {}
        self._ik_check_boxes: dict[str, str] = {}
        self._copy_position_check_boxes: dict[str, str] = {}
        self._copy_rotation_check_boxes: dict[str, str] = {}
        self.window_name = cmds.window(WINDOW_NAME, title='Retarget Animation From Joint to Controller',
                                       widthHeight=(642, 893))
        # Layouts
        main_layout = cmds.columnLayout(adjustableColumn=True)
        self.reload_button = cmds.button(parent=main_layout, visible=False, label='Reload')  # This is a debug control.
        cmds.text(parent=main_layout, label='Assign all required bones, '
                                            'make sure the character is in a t-pose, '
                                            'and then connect to Marionette.\n'
                                            'If using IK, then assign FK bones to '
                                            'the upper and lower IK bones in order for retargeting to function.')

        character_button_layout = cmds.formLayout(parent=main_layout)

        image_layout = cmds.rowLayout(parent=character_button_layout,
                                      numberOfColumns=len(image_paths))

        preset_layout = cmds.rowLayout(parent=main_layout,
                                       numberOfColumns=4,
                                       height=45,
                                       columnWidth=[(1, 225), (2, 185)])

        self._status_text = cmds.text(parent=main_layout, label='Status: Not connected.', align='center')

        additional_button_layout = cmds.rowLayout(parent=main_layout,
                                                  numberOfColumns=3,
                                                  height=45,
                                                  columnWidth=[(1, 225), (2, 185), (3, 100)])

        tab_layout = cmds.tabLayout(parent=main_layout, innerMarginWidth=5, innerMarginHeight=5)
        body_tab = cmds.columnLayout(parent=tab_layout)
        head_tab = cmds.columnLayout(parent=tab_layout)
        left_hand_tab = cmds.columnLayout(parent=tab_layout)
        right_hand_tab = cmds.columnLayout(parent=tab_layout)
        tabs = [body_tab, head_tab, left_hand_tab, right_hand_tab]
        cmds.tabLayout(tab_layout, edit=True, tabLabel=(
            (body_tab, 'Body'),
            (head_tab, 'Head'),
            (left_hand_tab, 'Left Hand'),
            (right_hand_tab, 'Right hand'),
        ))

        title_layouts: list[str] = []
        scroll_layouts: list[str] = []
        fields_layouts: list[str] = []
        empty_text_field_layouts: list[str] = []
        controller_text_field_layouts: list[str] = []
        translation_checkbox_layouts: list[str] = []
        rotation_checkbox_layouts: list[str] = []
        for i in range(4):
            title_layouts.append(cmds.rowLayout(parent=tabs[i],
                                                numberOfColumns=4,
                                                columnWidth=[(1, 100), (2, 275), (3, 75), (4, 75)]))

            scroll_layouts.append(cmds.scrollLayout(parent=tabs[i],
                                                    height=200,
                                                    width=635,
                                                    horizontalScrollBarThickness=16,
                                                    verticalScrollBarThickness=16))

            fields_layouts.append(cmds.rowLayout(parent=scroll_layouts[i],
                                                 numberOfColumns=4,
                                                 columnWidth=[(1, 100), (2, 305), (3, 75), (4, 75)]))

            empty_text_field_layouts.append(cmds.gridLayout(parent=fields_layouts[i],
                                                            numberOfRows=1,
                                                            numberOfColumns=1,
                                                            cellWidthHeight=(205, 25)))

            controller_text_field_layouts.append(cmds.gridLayout(parent=fields_layouts[i],
                                                                 numberOfRows=9,
                                                                 numberOfColumns=2,
                                                                 cellWidthHeight=(137, 25)))

            translation_checkbox_layouts.append(cmds.gridLayout(parent=fields_layouts[i],
                                                                numberOfRows=9,
                                                                numberOfColumns=1,
                                                                cellWidthHeight=(40, 25)))

            rotation_checkbox_layouts.append(cmds.gridLayout(parent=fields_layouts[i],
                                                             numberOfRows=9,
                                                             numberOfColumns=1,
                                                             cellWidthHeight=(40, 25)))

        # Create images
        for image_path in image_paths:
            cmds.picture(parent=image_layout,
                         image=image_path)

        # Create buttons
        for label, left, top in info_list:
            button = cmds.iconTextButton(parent=character_button_layout,
                                         label=label,
                                         width=20,
                                         height=20,
                                         command=lambda l=label: self.on_controller_button_clicked(l))
            self._controller_buttons[label] = button

            cmds.formLayout(character_button_layout,
                            edit=True,
                            attachForm=[(button, 'left', left), (button, 'top', top)])

        for label, left, top in ik_list:
            ik_check_box = cmds.checkBox(parent=character_button_layout,
                                         label='IK',
                                         value=False,
                                         changeCommand=lambda v, l=label: self.on_ik_check_box_changed(l, v))
            self._ik_check_boxes[label] = ik_check_box
            cmds.formLayout(character_button_layout,
                            edit=True,
                            attachForm=[(ik_check_box, 'left', left), (ik_check_box, 'top', top)])

        cmds.text(parent=preset_layout, label='')
        self.preset_option_menu = cmds.optionMenu(parent=preset_layout,
                                                  label='Preset',
                                                  width=160,
                                                  changeCommand=self.on_preset_option_menu_changed)
        for i in range(len(self.preset_storage)):
            cmds.menuItem(parent=self.preset_option_menu, label=f'Preset {i + 1}')
        cmds.menuItem(parent=self.preset_option_menu, label='<Unsaved Preset>')
        cmds.optionMenu(self.preset_option_menu, edit=True, select=len(self.preset_storage) + 1)
        self.save_or_delete_preset_button = cmds.button(parent=preset_layout, label='Save',
                                                        command=self.save_or_delete_preset)

        # Create empty space
        cmds.text(parent=additional_button_layout,
                  label='')

        # Create retarget button
        self.retarget_button = cmds.iconTextButton(parent=additional_button_layout,
                                                   style='textOnly',
                                                   label='Connect',
                                                   font='boldLabelFont',
                                                   backgroundColor=(0.4, 0.4, 0.4),
                                                   width=175,
                                                   height=30)

        # Create checkbox
        self.mirrored_select_checkbox = cmds.checkBox(parent=additional_button_layout,
                                                      label='Mirror Selection',
                                                      annotation='Select both left and right bones at the same times when using the bone picker',
                                                      value=True)

        # Create title and text for empty space
        # Title
        for i in range(4):
            cmds.text(parent=title_layouts[i],
                      label='',
                      width=100,
                      height=20)
            # Text
            cmds.text(parent=empty_text_field_layouts[i],
                      label='')

            # Create title, labels, and textfields for controllers
            # Title
            cmds.text(parent=title_layouts[i],
                      label='Controllers',
                      align='left',
                      width=275,
                      height=20,
                      font='boldLabelFont',
                      backgroundColor=(0.4, 0.4, 0.4))
            for label in tab_bones[i]:
                # labels
                cmds.text(parent=controller_text_field_layouts[i],
                          label=f'{label}:',
                          align='left')
                # textFields
                controller_text_field = cmds.textField(parent=controller_text_field_layouts[i],
                                                       placeholderText='Enter controller',
                                                       textChangedCommand=lambda v: self.on_preset_modified())
                self._controller_text_fields[label] = controller_text_field

            # Create title and checkboxes for translation
            # Title
            cmds.text(parent=title_layouts[i],
                      label='Translation',
                      align='left',
                      width=75,
                      height=20,
                      font='boldLabelFont',
                      backgroundColor=(0.4, 0.4, 0.4))
            for label in tab_bones[i]:
                # Checkboxes
                self._copy_position_check_boxes[label] = cmds.checkBox(parent=translation_checkbox_layouts[i],
                                                                       label='',
                                                                       value=True,
                                                                       changeCommand=lambda
                                                                           v: self.on_preset_modified())

            # Create title and checkboxes for rotation
            # Title
            cmds.text(parent=title_layouts[i],
                      label='Rotation',
                      align='left',
                      width=75,
                      height=20,
                      font='boldLabelFont',
                      backgroundColor=(0.4, 0.4, 0.4))
            for label in tab_bones[i]:
                # Checkboxes
                self._copy_rotation_check_boxes[label] = cmds.checkBox(parent=rotation_checkbox_layouts[i],
                                                                       label='',
                                                                       value=True,
                                                                       changeCommand=lambda
                                                                           v: self.on_preset_modified())

        self.on_preset_modified()
        self.auto_select_preset()
        # Display the window
        cmds.showWindow()

    def on_controller_button_clicked(self, label: str) -> None:
        selection = cmds.ls(selection=True)
        selection = selection[0] if selection else ''
        current_value = cmds.textField(self._controller_text_fields[label], query=True, text=True)
        has_twin = label in twin_labels and cmds.checkBox(self.mirrored_select_checkbox, query=True, value=True)

        if current_value == '':
            if selection == '':
                cmds.inViewMessage(assistMessage='No object selected', pos='midCenter', fade=True)
                return
            cmds.textField(self._controller_text_fields[label], edit=True, text=selection)
            if has_twin:
                twin = get_twin_name(selection)
                if twin is not None and cmds.objExists(twin):
                    cmds.textField(self._controller_text_fields[twin_labels[label]], edit=True, text=twin)
        else:
            cmds.textField(self._controller_text_fields[label], edit=True, text='')
            if has_twin:
                cmds.textField(self._controller_text_fields[twin_labels[label]], edit=True, text='')
        self.on_preset_modified()

    def on_preset_option_menu_changed(self, value: str) -> None:
        selected_option = cmds.optionMenu(self.preset_option_menu, query=True, select=True) - 1
        if selected_option == len(self.preset_storage):
            return
        self.selected_preset = selected_option
        self.on_selected_preset_changed()

    def on_ik_check_box_changed(self, label: str, value: bool) -> None:
        bones = {
            'LeftLegBendGoal': ['LeftLowerLeg', 'LeftUpperLeg'],
            'RightLegBendGoal': ['RightLowerLeg', 'RightUpperLeg'],
            'LeftArmBendGoal': ['LeftLowerArm', 'LeftUpperArm'],
            'RightArmBendGoal': ['RightLowerArm', 'RightUpperArm'],
        }
        for bone in bones[label]:
            if value:
                cmds.checkBox(self._copy_position_check_boxes[bone], edit=True, value=False)
                cmds.checkBox(self._copy_rotation_check_boxes[bone], edit=True, value=False)
            else:
                cmds.checkBox(self._copy_position_check_boxes[bone], edit=True, value=True)
                cmds.checkBox(self._copy_rotation_check_boxes[bone], edit=True, value=True)
        self.on_preset_modified()

    def on_preset_modified(self) -> None:
        if self.selected_preset_is_changing:
            return
        self.selected_preset = None
        self.on_selected_preset_changed()

    def on_selected_preset_changed(self) -> None:
        if self.selected_preset is not None:
            self.selected_preset_is_changing = True
            preset = self.preset_storage[self.selected_preset]
            for human_name, bone_name, copy_rotation, copy_position in zip(preset.human_names, preset.bone_names,
                                                                           preset.copy_rotation, preset.copy_position):
                cmds.textField(self._controller_text_fields[human_name], edit=True, text=bone_name)
                cmds.checkBox(self._copy_position_check_boxes[human_name], edit=True, value=copy_position)
                cmds.checkBox(self._copy_rotation_check_boxes[human_name], edit=True, value=copy_rotation)
            for check_box, enabled in zip(self._ik_check_boxes.values(), preset.enable_ik):
                cmds.checkBox(check_box, edit=True, value=enabled)
            self.selected_preset_is_changing = False
        self.redraw_preset_options()
        self.update_controller_icons()

    def auto_select_preset(self) -> None:
        for i, preset in enumerate(self.preset_storage):
            preset_is_valid = True
            for bone_name in preset.bone_names:
                if bone_name and not cmds.objExists(bone_name):
                    preset_is_valid = False
                    break
            if preset_is_valid:
                self.selected_preset = i
                self.on_selected_preset_changed()
                break

    def update_controller_icons(self) -> None:
        assigned_count = {}
        for label, _, _ in info_list:
            value = cmds.textField(self._controller_text_fields[label], query=True, text=True).strip()
            if value in assigned_count:
                assigned_count[value] += 1
            else:
                assigned_count[value] = 1
        for label, _, _ in info_list:
            value = cmds.textField(self._controller_text_fields[label], query=True, text=True).strip()
            annotation = f'{label}: {value}'
            if value == '':
                if label in required_bones:
                    image = required_empty_icon_name
                    annotation = f'{label}: Required joint'
                else:
                    image = optional_empty_icon_name
                    annotation = f'{label}: Optional joint'
            else:
                if assigned_count[value] > 1:
                    annotation += f" (Error: '{value}' has been assigned in multiple places)"
                    image = filled_error_icon_name
                elif not cmds.objExists(value):
                    annotation += f" (Error: Joint with name '{value}' not found)"
                    image = filled_error_icon_name
                else:
                    image = filled_ok_icon_name
            cmds.iconTextButton(self._controller_buttons[label], edit=True, image=image, annotation=annotation)

            if label in self._ik_check_boxes:
                enabled = cmds.checkBox(self._ik_check_boxes[label], query=True, value=True)
                cmds.iconTextButton(self._controller_buttons[label], edit=True, visible=enabled)
                cmds.textField(self._controller_text_fields[label], edit=True, enable=enabled)
                cmds.checkBox(self._copy_position_check_boxes[label], edit=True, enable=enabled)
                cmds.checkBox(self._copy_rotation_check_boxes[label], edit=True, enable=enabled)

    def enable_mapping(self) -> None:
        for text_field in self._controller_text_fields.values():
            cmds.textField(text_field, edit=True, enable=True)
        for button in self._controller_buttons.values():
            cmds.iconTextButton(button, edit=True, enable=True)
        for check_box in self._copy_position_check_boxes.values():
            cmds.checkBox(check_box, edit=True, enable=True)
        for check_box in self._copy_rotation_check_boxes.values():
            cmds.checkBox(check_box, edit=True, enable=True)
        for check_box in self._ik_check_boxes.values():
            cmds.checkBox(check_box, edit=True, enable=True)
        cmds.optionMenu(self.preset_option_menu, edit=True, enable=True)
        cmds.button(self.save_or_delete_preset_button, edit=True, enable=True)

    def disable_mapping(self) -> None:
        for text_field in self._controller_text_fields.values():
            cmds.textField(text_field, edit=True, enable=False)
        for button in self._controller_buttons.values():
            cmds.iconTextButton(button, edit=True, enable=False)
        for check_box in self._copy_position_check_boxes.values():
            cmds.checkBox(check_box, edit=True, enable=False)
        for check_box in self._copy_rotation_check_boxes.values():
            cmds.checkBox(check_box, edit=True, enable=False)
        for check_box in self._ik_check_boxes.values():
            cmds.checkBox(check_box, edit=True, enable=False)
        cmds.optionMenu(self.preset_option_menu, edit=True, enable=False)
        cmds.button(self.save_or_delete_preset_button, edit=True, enable=False)

    def preset_from_ui(self) -> Preset:
        mappings = {}
        for label, _, _ in info_list:
            assigned_value = cmds.textField(self._controller_text_fields[label], query=True, text=True).strip()
            if assigned_value == '':
                continue
            if label in self._ik_check_boxes and not cmds.checkBox(self._ik_check_boxes[label], query=True, value=True):
                continue
            copy_rotation = cmds.checkBox(self._copy_rotation_check_boxes[label], query=True, value=True)
            copy_position = cmds.checkBox(self._copy_position_check_boxes[label], query=True, value=True)
            mappings[label] = (label, assigned_value, copy_rotation, copy_position)

        sorted_mappings = []
        for label in reversed(mapping_order):
            if label in mappings:
                sorted_mappings.append(mappings[label])

        enable_ik = [cmds.checkBox(box, query=True, value=True) for box in self._ik_check_boxes.values()]
        return Preset.from_mappings(sorted_mappings, enable_ik)

    def validate_mappings(self) -> bool:
        assigned_count = {}
        for label, _, _ in info_list:
            value = cmds.textField(self._controller_text_fields[label], query=True, text=True).strip()
            if value == '':
                continue
            if value in assigned_count:
                assigned_count[value] += 1
            else:
                assigned_count[value] = 1

        error_message = ''
        missing_bones = False
        for label, _, _ in info_list:
            value = cmds.textField(self._controller_text_fields[label], query=True, text=True).strip()
            if value == '':
                if label in required_bones:
                    error_message = f'Error: Missing required joint {label}'
                    missing_bones = True
                    break
            else:
                if assigned_count[value] > 1:
                    error_message = f"Error: '{value}' has been assigned in multiple places"
                    break
                elif not cmds.objExists(value):
                    error_message = f"Error: Unable to find '{value}' assigned to {label}"
                    break
        if error_message == '':
            return True
        else:
            if missing_bones:
                choice = cmds.confirmDialog(
                    title='Warning!',
                    message='One or more required joints are missing. Do you still want to connect?',
                    button=['Connect', 'Cancel'],
                    defaultButton='Connect',
                    cancelButton='Cancel',
                    dismissString='Cancel'
                )
                if choice == 'Connect':
                    return True
            cmds.inViewMessage(assistMessage=error_message, pos='midCenter', fade=True)
            return False

    def set_status(self, msg: str) -> None:
        cmds.text(self._status_text, edit=True, label=f'Status: {msg}')

    def save_or_delete_preset(self, _) -> None:
        if self.selected_preset is None:
            mappings = []
            for label, _, _ in info_list:
                assigned_value = cmds.textField(self._controller_text_fields[label], query=True, text=True).strip()
                copy_rotation = cmds.checkBox(self._copy_rotation_check_boxes[label], query=True, value=True)
                copy_position = cmds.checkBox(self._copy_position_check_boxes[label], query=True, value=True)
                mappings.append((label, assigned_value, copy_rotation, copy_position))
            enable_ik = [cmds.checkBox(box, query=True, value=True) for box in self._ik_check_boxes.values()]

            self.preset_storage.append(Preset.from_mappings(mappings, enable_ik))
            self.selected_preset = len(self.preset_storage) - 1
            self.redraw_preset_options()
        else:
            del self.preset_storage[self.selected_preset]
            self.selected_preset = None
            self.redraw_preset_options()

    def redraw_preset_options(self) -> None:
        cmds.optionMenu(self.preset_option_menu, edit=True, deleteAllItems=True)
        for i in range(len(self.preset_storage)):
            cmds.menuItem(parent=self.preset_option_menu, label=f'Preset {i + 1}')
        if self.selected_preset is None:
            cmds.menuItem(parent=self.preset_option_menu, label='<Unsaved Preset>')
        if self.selected_preset is None:
            cmds.optionMenu(self.preset_option_menu, edit=True, select=len(self.preset_storage) + 1)
            cmds.button(self.save_or_delete_preset_button, edit=True, label='Save')
        else:
            cmds.optionMenu(self.preset_option_menu, edit=True, select=self.selected_preset + 1)
            cmds.button(self.save_or_delete_preset_button, edit=True, label='Delete')


if __name__ == '__main__':
    MarionetteWindow()
