import copy
from dataclasses import dataclass, field
from typing import Iterator, List

from maya import cmds


@dataclass
class Preset:
    human_names: List[str]
    bone_names: List[str]
    copy_rotation: List[bool]
    copy_position: List[bool]
    enable_ik: List[bool]

    def __deepcopy__(self, memo={}) -> 'Preset':
        return Preset(
            human_names=copy.deepcopy(self.human_names, memo),
            bone_names=copy.deepcopy(self.bone_names, memo),
            copy_rotation=copy.deepcopy(self.copy_rotation, memo),
            copy_position=copy.deepcopy(self.copy_position, memo),
            enable_ik=copy.deepcopy(self.enable_ik, memo),
        )

    @staticmethod
    def from_mappings(mappings: List[tuple[str, str, bool, bool]], enable_ik: List[bool]) -> 'Preset':
        return Preset(
            human_names=[human_name for human_name, bone_name, copy_rotation, copy_position in mappings],
            bone_names=[bone_name for human_name, bone_name, copy_rotation, copy_position in mappings],
            copy_rotation=[copy_rotation for human_name, bone_name, copy_rotation, copy_position in mappings],
            copy_position=[copy_position for human_name, bone_name, copy_rotation, copy_position in mappings],
            enable_ik=enable_ik
        )


class PresetStorage:
    def __getitem__(self, i: int) -> Preset:
        assert i < len(self)
        assert cmds.optionVar(exists=f'marionette.presets[{i}].human_names')
        assert cmds.optionVar(exists=f'marionette.presets[{i}].bone_names')
        assert cmds.optionVar(exists=f'marionette.presets[{i}].copy_rotation')
        assert cmds.optionVar(exists=f'marionette.presets[{i}].copy_position')

        return Preset(
            human_names=cmds.optionVar(q=f'marionette.presets[{i}].human_names'),
            bone_names=cmds.optionVar(q=f'marionette.presets[{i}].bone_names'),
            copy_rotation=[c != 0 for c in cmds.optionVar(q=f'marionette.presets[{i}].copy_rotation')],
            copy_position=[c != 0 for c in cmds.optionVar(q=f'marionette.presets[{i}].copy_position')],
            enable_ik=[c != 0 for c in cmds.optionVar(q=f'marionette.presets[{i}].enable_ik')]
            if cmds.optionVar(exists=f'marionette.presets[{i}].enable_ik')
            # Upgrade older version of the plugin.
            else [self._has_bone(i, label) for label in
                  ['LeftArmBendGoal', 'RightArmBendGoal', 'LeftLegBendGoal', 'RightLegBendGoal']],
        )

    def _has_bone(self, preset_index: int, label: str) -> bool:
        # Only used for upgrading.
        human_names = cmds.optionVar(q=f'marionette.presets[{preset_index}].human_names')
        bone_names = cmds.optionVar(q=f'marionette.presets[{preset_index}].bone_names')
        for human_name, bone_name in zip(human_names, bone_names):
            if human_name == label and bone_name != '':
                return True
        else:
            return False

    def __len__(self) -> int:
        if cmds.optionVar(exists='marionette.presets.length'):
            return cmds.optionVar(q='marionette.presets.length')
        else:
            return 0

    def __iter__(self) -> Iterator[Preset]:
        for i in range(len(self)):
            yield self[i]

    def append(self, preset: Preset) -> None:
        i = len(self)
        cmds.optionVar(stringArray=f'marionette.presets[{i}].human_names')
        cmds.optionVar(stringArray=f'marionette.presets[{i}].bone_names')
        cmds.optionVar(intArray=f'marionette.presets[{i}].copy_rotation')
        cmds.optionVar(intArray=f'marionette.presets[{i}].copy_position')
        for human_name in preset.human_names:
            cmds.optionVar(stringValueAppend=[f'marionette.presets[{i}].human_names', human_name])
        for bone_name in preset.bone_names:
            cmds.optionVar(stringValueAppend=[f'marionette.presets[{i}].bone_names', bone_name])
        for copy_rotation in preset.copy_rotation:
            cmds.optionVar(intValueAppend=[f'marionette.presets[{i}].copy_rotation', 1 if copy_rotation else 0])
        for copy_position in preset.copy_position:
            cmds.optionVar(intValueAppend=[f'marionette.presets[{i}].copy_position', 1 if copy_position else 0])
        for enable_ik in preset.enable_ik:
            cmds.optionVar(intValueAppend=[f'marionette.presets[{i}].enable_ik', 1 if enable_ik else 0])
        cmds.optionVar(intValue=['marionette.presets.length', len(self) + 1])

    def __delitem__(self, i: int):
        for j in range(i, len(self) - 1):
            cmds.optionVar(stringArray=f'marionette.presets[{j}].human_names')
            cmds.optionVar(stringArray=f'marionette.presets[{j}].bone_names')
            cmds.optionVar(intArray=f'marionette.presets[{j}].copy_rotation')
            cmds.optionVar(intArray=f'marionette.presets[{j}].copy_position')

            for human_name in self[j + 1].human_names:
                cmds.optionVar(stringValueAppend=[f'marionette.presets[{i}].human_names', human_name])
            for bone_name in self[j + 1].bone_names:
                cmds.optionVar(stringValueAppend=[f'marionette.presets[{i}].bone_names', bone_name])
            for copy_rotation in self[j + 1].copy_rotation:
                cmds.optionVar(intValueAppend=[f'marionette.presets[{i}].copy_rotation', 1 if copy_rotation else 0])
            for copy_position in self[j + 1].copy_position:
                cmds.optionVar(intValueAppend=[f'marionette.presets[{i}].copy_position', 1 if copy_position else 0])
            for enable_ik in self[j + 1].enable_ik:
                cmds.optionVar(intValueAppend=[f'marionette.presets[{i}].enable_ik', 1 if enable_ik else 0])

        cmds.optionVar(remove=f'marionette.presets[{len(self) - 1}].human_names')
        cmds.optionVar(remove=f'marionette.presets[{len(self) - 1}].bone_names')
        cmds.optionVar(remove=f'marionette.presets[{len(self) - 1}].copy_rotation')
        cmds.optionVar(remove=f'marionette.presets[{len(self) - 1}].copy_position')
        cmds.optionVar(remove=f'marionette.presets[{len(self) - 1}].enable_ik')

        cmds.optionVar(intValue=['marionette.presets.length', len(self) - 1])
