from maya import cmds

from marionette.client import Client
from marionette.marionette_window import MarionetteWindow, WINDOW_NAME
from marionette.message_handler import MessageHandler
from marionette.reload_package import unload_packages


def show_window() -> None:
    if cmds.window(WINDOW_NAME, exists=True):
        cmds.showWindow(WINDOW_NAME)
        return

    window = MarionetteWindow()

    def reload() -> None:
        client.close()
        cmds.deleteUI(window.window_name)
        unload_packages(packages=['marionette'])
        import marionette
        marionette.show_window()

    def on_connect_button_pressed() -> None:
        if client._wants_to_connect:
            client.stop()
            cmds.iconTextButton(window.retarget_button, edit=True, enable=False, label="Disconnecting...")
        else:
            if window.validate_mappings():
                window.disable_mapping()
                client.start("127.0.0.1", 11183, window.preset_from_ui())
                cmds.iconTextButton(window.retarget_button, edit=True, enable=True, label="Cancel")
        window.update_controller_icons()

    def on_connection_attempt_started() -> None:
        if not cmds.window(window.window_name, query=True, exists=True):
            return
        window.set_status("Connecting...")

    def on_connected() -> None:
        if not cmds.window(window.window_name, query=True, exists=True):
            return
        window.set_status("Configuring connection...")

    def on_disconnected() -> None:
        if not cmds.window(window.window_name, query=True, exists=True):
            return
        window.set_status("Not connected.")
        if not client._wants_to_connect:
            cmds.iconTextButton(window.retarget_button, edit=True, enable=True, label="Connect")
            window.enable_mapping()

    def on_set_up_completed() -> None:
        if not cmds.window(window.window_name, query=True, exists=True):
            return
        window.set_status("Connected.")

    client = Client(MessageHandler(
        connection_attempt_started=on_connection_attempt_started,
        connected=on_connected,
        disconnected=on_disconnected,
        set_up_completed=on_set_up_completed
    ))
    cmds.window(window.window_name, edit=True, closeCommand=lambda: client.close())
    # cmds.button(window.reload_button, edit=True, visible=True, command=lambda _: reload())
    cmds.iconTextButton(window.retarget_button, edit=True, command=on_connect_button_pressed)
    # on_connect_button_pressed()
